// upload_and_save_doc
//
// @(#)main.go  星期四, 五月 30, 2024
// Copyright(c) 2024, reinhold@Tencent. All rights reserved.

package main

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"path/filepath"
	"strconv"
	"strings"
	"time"

	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	tchttp "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/http"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
	"github.com/tencentyun/cos-go-sdk-v5"
)

type DescribeStorageCredential struct {
	Response struct {
		Bucket      string `json:"Bucket"`
		CorpUin     string `json:"CorpUin"`
		Credentials struct {
			TmpSecretId  string `json:"TmpSecretId"`
			TmpSecretKey string `json:"TmpSecretKey"`
			Token        string `json:"Token"`
		} `json:"Credentials"`
		ExpiredTime int    `json:"ExpiredTime"`
		FilePath    string `json:"FilePath"`
		ImagePath   string `json:"ImagePath"`
		Region      string `json:"Region"`
		RequestId   string `json:"RequestId"`
		StartTime   int    `json:"StartTime"`
		Type        string `json:"Type"`
		UploadPath  string `json:"UploadPath"`
	} `json:"Response"`
}

const (
	// 线上环境
	EndPoint        = "lke.tencentcloudapi.com"
	SecretID        = "xxxxxxxxxxxxxxxxxx"
	SecretKey       = "xxxxxxxxxxxxxxxxxx"
	BotBizID        = "xxxxxxxxxxxxxxxxxx"
	TypeKeyRealtime = "realtime" // 实时文件上传类型
	TypeKeyOffline  = "offline"  // 离线文档上传类型
)

func main() {

	filePath := "./test.txt"

	// 获取文件名
	fileName := filepath.Base(filePath)
	fmt.Println("文件名:", fileName)

	// 获取文件后缀
	fileExt := strings.Trim(filepath.Ext(filePath), ".")
	fmt.Println("文件后缀:", fileExt)

	// 1. 获取临时密钥，注意该密钥有过期时间显示； 需要传入文件类型才能获取上传权限
	// 请注意获取临时必要的时候需要带上文件后缀
	tmpSecretId, tmpSecretKey, token, uploadPath, bucketURL, err := getDescribeStorageCredential(fileExt)
	if err != nil {
		fmt.Printf("\ngetDescribeStorageCredential err:%+v", err)
		return
	}
	fmt.Printf(" \n ======= 1.getDescribeStorageCredential success =====\n")
	fmt.Printf("\ntemSecretID:%s,tmpSecretKey:%s,token:%s,uploadPath:%s,bucketURL:%s\n\n", tmpSecretId, tmpSecretKey, token, uploadPath, bucketURL)

	// 2. 将文件上传到cos [此方法只适用于公有云]
	eTag, cosHash, fileSize, err := cosUploadFile(tmpSecretId, tmpSecretKey, token, uploadPath, bucketURL, filePath)
	if err != nil {
		fmt.Printf("\ncosUploadFile err:%+v", err)
		return
	}
	fmt.Printf(" \n ======= 2.cosUploadFile success =====\n")
	fmt.Printf("eTag:%s,cosHash:%s,fileSize:%d\n", eTag, cosHash, fileSize)

	// 3. saveDoc [打开外部访问链接]
	saveDocReq := fmt.Sprintf(`{
					"BotBizId": "%s",
					"FileName": "%s",
					"FileType": "%s",
					"CosUrl": "%s",
					"ETag": %s,
					"CosHash": "%s",
					"Size": "%d",
					"AttrRange": 1,
					"Source": 0,
					"WebUrl": "",
					"AttrLabels": [],
					"IsRefer": true,
					"ReferUrlType": 0,
					"ExpireStart": "%d",
					"Opt": 2
	     }`, BotBizID, fileName, fileExt, uploadPath, eTag, cosHash, fileSize, time.Now().Unix())
	_, err = commonReq("SaveDoc", saveDocReq, "POST")
	if err != nil {
		fmt.Printf("ERROR: SaveDoc Failed! req:%+v,err:%+v", saveDocReq, err)
		return
	}

	fmt.Printf(" \n ======= 3.SaveDoc success =====\n")

	fmt.Println(" \n\n========== done ===========\n\n")
}

func getDescribeStorageCredential(fileExt string) (TmpSecretId, TmpSecretKey, Token, UploadPath, bucketURL string, err error) {
	//  1 获取证书
	// 请注意，此处为离线文档上传，TypeKey取值为offline; 如果需要复用此处代码上传实时文档，需要修改TypeKey取值为 realtime
	storageReq := fmt.Sprintf(`{ "BotBizId":"%s" ,"FileType":"%s","TypeKey":"%s","IsPublic": false}`, BotBizID, fileExt, TypeKeyOffline)
	fmt.Printf("getDescribeStorageCredential req:%+v", storageReq)
	resp, err := commonReq("DescribeStorageCredential", storageReq, "POST")
	if err != nil {
		return
	}

	var storageResp DescribeStorageCredential
	json.Unmarshal([]byte(resp), &storageResp)

	bucketURL = "https://" + storageResp.Response.Bucket + "." + storageResp.Response.Type + "." + storageResp.Response.Region + ".myqcloud.com"

	return storageResp.Response.Credentials.TmpSecretId, storageResp.Response.Credentials.TmpSecretKey, storageResp.Response.Credentials.Token, storageResp.Response.UploadPath, bucketURL, nil
}

func cosUploadFile(tmpSecretId, tmpSecretKey, token, uploadPath, bucketURL, filePath string) (eTag, cosHash string, fileSize int, err error) {
	//https://console.tencentcloud/api/explorer?Product=cos&Version=2018-11-26&Action=PutObject
	u, _ := url.Parse(bucketURL)
	b := &cos.BaseURL{BucketURL: u}
	client := cos.NewClient(b, &http.Client{
		Transport: &cos.AuthorizationTransport{
			SecretID:     tmpSecretId,
			SecretKey:    tmpSecretKey,
			SessionToken: token,
		},
	})

	// Case1 使用 Put 上传对象
	opt := &cos.ObjectPutOptions{
		ObjectPutHeaderOptions: &cos.ObjectPutHeaderOptions{
			// ContentType: "text/html", // 此处要么不填写，要么根据具体上传的文件类型来传参
		},
	}

	response, err := client.Object.PutFromFile(context.Background(), uploadPath, filePath, opt)
	if err != nil {
		fmt.Printf("\nERROR: cosUploadFle Failed! err:%+v", err)
		return
	}

	content, err := ioutil.ReadFile(filePath)
	eTag = response.Header.Get("Etag")
	cosHash = response.Header.Get("X-Cos-Hash-Crc64ecma")

	// 需要注意etag要是这种样式的：\"0fe54d66fd5d4c6b52a87b8ef23837d9\"'
	escapedETag := strconv.Quote(eTag)

	return escapedETag, cosHash, len(content), nil

}

func commonReq(action, params, reqMethod string) (string, error) {
	credential := common.NewCredential(SecretID, SecretKey)
	cpf := profile.NewClientProfile()
	cpf.HttpProfile.Endpoint = EndPoint
	cpf.HttpProfile.ReqMethod = reqMethod
	client := common.NewCommonClient(credential, "ap-jakarta", cpf).WithLogger(log.Default())

	request := tchttp.NewCommonRequest("lke", "2023-11-30", action)

	err := request.SetActionParameters(params)
	if err != nil {
		fmt.Println(err)
		return "", err
	}

	response := tchttp.NewCommonResponse()
	err = client.Send(request, response)
	if err != nil {
		if strings.Contains(err.Error(), "450027-没有可发布的文档/问答/拒答/任务流程/应用配置") {
			return "", nil
		}
		if strings.Contains(err.Error(), "450020-文档已存在") {
			return "", err
		}
		fmt.Printf("ERROR: Failed Invoke API:action:%s,params:%s,reqMethod:%s,err:%+v", action, params, reqMethod, err)
		return "", err
	}

	//log.Printf("Action:%s,resp:%s", action, response.GetBody())

	return string(response.GetBody()), nil

}
